<?php

namespace Velis\User;

use ArrayIterator;
use ArrayObject;
use Velis\App;
use Velis\Lang;

//@codingStandardsIgnoreStart
if (App::$config->anonymize->enabled) {
    /**
     * GDPR Anonymization
     * @author Olek Procki <olo@velis.pl>
     */
    trait AnonymizeTrait
    {
        /**
         * Fields to be replaced when anonymization is enabled
         * @var array
         */
        protected static $_anonymizedFields = [
            'first_name'            => '{USER_NAME}-{$id}',
            'last_name'             => '{USER_SURNAME}-{$id}',
            'name'                  => '{USER_NAME}-{$id}',
            'email'                 => 'bcc@velis.pl',
            'email_address'         => 'bcc@velis.pl',
            'search_name'           => '{USER_NAME} {USER_SURNAME}-{$id}',
            'owner_user_name'       => '{USER_NAME} {USER_SURNAME}',
            'request_user'          => '{USER_NAME} {USER_SURNAME}',
            'responsible_user_name' => '{USER_NAME} {USER_SURNAME}',
            'identity_doc_no'       => 'ID',
            'has_avatar'            => '0',
            'phone_no'              => '000-000-000',
            'mobile_phone_no'       => '000000000',
        ];


        /**
         * ArrayAccess get override
         *
         * @param string $key
         * @return string
         */
        public function &offsetGet($key)
        {
            if (App::$config->anonymize->enabled &&
                array_key_exists($key, self::$_anonymizedFields)
            ) {
                $anonymizedData = $this->getAnonymizedData($key);

                return $anonymizedData;
            }

            $value = parent::offsetGet($key);

            if (is_array($value) && !in_array($key, ["building_complex_ids", "substitute_for", "acl_roles"])) {
                $value = $this[$key] = new ArrayObject($value);

                return $value;
            }

            return $value;
        }


        /**
         * Magic getter override
         *
         * @param string $key
         * @return string
         */
        public function __get($key)
        {
            if (App::$config->anonymize->enabled &&
                array_key_exists($key, self::$_anonymizedFields)) {
                return $this->getAnonymizedData($key);
            }
            return parent::__get($key);
        }


        /**
         * Array copy override
         * @return array
         */
        public function getArrayCopy()
        {
            if (!App::$config->anonymize->enabled) {
                return parent::getArrayCopy();
            }
            $copy = parent::getArrayCopy();

            foreach ($copy as $field => $value) {
                if (array_key_exists($field, self::$_anonymizedFields)) {
                    $copy[$field] = $this->getAnonymizedData($field);
                }
            }
            return $copy;
        }


        /**
         * Returns iterator object with anonymized data
         * @return ArrayIterator
         */
        public function getIterator()
        {
            if (App::$config->anonymize->enabled) {
                return new ArrayIterator($this->getArrayCopy());
            }
            return parent::getIterator();
        }


        /**
         * Returns anonymized data for user/person
         *
         * @param string $field
         * @return string
         */
        public function getAnonymizedData($field)
        {
            if (is_array($this->_getPrimaryKeyField())) {
                $itemId = $this['user_id'] ?: $this['person_id'] ?: '';
            } else {
                $itemId = $this->id();
            }

            $langs = [
                '{USER_NAME}',
                '{USER_SURNAME}'
            ];
            if (Lang::getLanguage() == 'pl') {
                $replacements = [
                    'Imię',
                    'Nazwisko'
                ];
            } else {
                $replacements = [
                    'Name',
                    'LastName'
                ];
            }

            return str_replace(
                $langs,
                $replacements,
                str_replace('{$id}', $itemId, self::$_anonymizedFields[$field])
            );
        }


        /**
         * Returns sanitized data with personal data removed as well
         *
         * @param bool $stash
         * @return static
         */
        public function sanitize($stash = true)
        {
            if (App::$config->anonymize->enabled) {
                foreach (parent::getArrayCopy() as $key => $value) {
                    if (array_key_exists($key, self::$_anonymizedFields)) {
                        $this->_stashedFields[$key] = parent::offsetGet($key);
                        parent::offsetUnset($key);
                    }
                }
            }
            return parent::sanitize($stash);
        }
    }
} else {
    trait AnonymizeTrait 
    {
        // do nothing
    }   
}
//@codingStandardsIgnoreEnd
