<?php

namespace Velis;

use Netcarver\Textile\Parser;

/**
 * Modified Textile class for Velis usage :D
 *
 * @author Robert Jamróz <robert.jamroz@velis.pl>
 */
class Textile extends Parser
{
    /**
     * @var string Raw quote filter
     */
    const QUOTE_RAW = 'raw';


    /**
     * @var string Stripped quote filter
     */
    const QUOTE_STRIPPED = 'stripped';


    /**
     * Parses the given Textile input in restricted mode.
     *
     * This method should be used for any untrusted user input,
     * including comments or forum posts.
     *
     * This method escapes any raw HTML input, ignores unsafe
     * attributes, links only whitelisted URL schemes
     * and by default also prevents the use of images and
     * extra Textile formatting, accepting only paragraphs
     * and blockquotes as valid block tags.
     *
     * <code>
     * $parser = new \Netcarver\Textile\Parser();
     * echo $parser->textileRestricted('h1. Hello World!');
     * </code>
     *
     * @param  string $text    The Textile input to parse
     * @param  bool   $lite    Controls lite mode, allowing extra formatting
     * @param  bool   $noimage Allow images
     * @param  string $rel     Relationship attribute applied to generated links
     * @return string Parsed $text
     * @see    Parser::textileThis()
     * @api
     */
    public function textileRestricted($text, $lite = true, $noimage = true, $rel = 'nofollow')
    {
        $this
            ->setLite($lite)
            ->setImages(!$noimage)
            ->setLinkRelationShip($rel)
            ->prepare()
        ;

        $this->url_schemes = $this->restricted_url_schemes;
        $this->restricted = true;

        $text = $this->quotes($text);

        // Make html
        $html = $this->parse($text);

        preg_match_all("|<li>|", $html, $liOpen, PREG_PATTERN_ORDER);
        preg_match_all("|</li>|", $html, $liClosed, PREG_PATTERN_ORDER);

        if (count($liOpen[0]) == count($liClosed[0])) {
            return $html;
        } else {
            return $text;
        }
    }


    /**
     * @param string $text
     * @return string
     */
    public function quotes($text)
    {
        $text = explode("\n", $text);
        $out = [];

        $hasBq = false;

        foreach ($text as $lineNo => $line) {
            $line = trim($line);
            if (strlen($line) && $line[0] == '>') {
                if ($hasBq) {
                    $line = ltrim($line, '>');
                } else {
                    if ($lineNo) {
                        $out[] = '';
                    }
                    $line = 'bq.. ' . ltrim($line, '>');
                    $hasBq = true;
                }
            } else {
                $line = $text[$lineNo];
                if ($lineNo) {
                    $prevLine = trim($text[$lineNo - 1]);
                    if (strlen($prevLine) && $prevLine[0] == '>') {
                        $out[] = '';
                        $out[] = 'p. ';
                        $out[] = '';
                    }
                }
                $hasBq = false;
            }
            $out[] = $line;
        }

        return join("\n", $out);
    }
}
