<?php

namespace Velis\Test\MaintenanceAlert;

use DateTimeImmutable;
use Generator;
use PHPUnit\Framework\TestCase;
use Velis\MaintenanceAlert\MaintenanceAlertFormatter;

/**
 * @author Szymon Janaczek <szymon.janaczek@velistech.com>
 */
class MaintenanceAlertFormatterTest extends TestCase
{
    private const DEFAULT_TEMPLATE = 'Maintenance from XXX to YYY';

    /**
     * @dataProvider provideFormatTestCases
     * @throws \Exception
     */
    public function testFormat(
        string $startDate,
        string $stopDate,
        string $expectedResult,
        string $template = self::DEFAULT_TEMPLATE,
        string $fromPlaceholder = 'XXX',
        string $toPlaceholder = 'YYY',
    ): void {
        $formatter = new MaintenanceAlertFormatter($template, $fromPlaceholder, $toPlaceholder);
        $start = new DateTimeImmutable($startDate);
        $stop = new DateTimeImmutable($stopDate);

        $result = $formatter->format($start, $stop);

        $this->assertEquals($expectedResult, $result);
    }

    public function provideFormatTestCases(): Generator
    {
        yield 'same day' => [
            '2024-07-30 10:00:00',
            '2024-07-30 14:00:00',
            'Maintenance from 2024-07-30 10:00 to 14:00',
        ];

        yield 'different days' => [
            '2024-07-30 22:00:00',
            '2024-07-31 02:00:00',
            'Maintenance from 2024-07-30 22:00 to 2024-07-31 02:00',
        ];

        yield 'custom placeholders' => [
            '2024-07-30 10:00:00',
            '2024-07-30 14:00:00',
            'Maintenance 2024-07-30 10:00 to 14:00',
            'Maintenance [FROM] to [TO]',
            '[FROM]',
            '[TO]',
        ];

        yield 'midnight transition' => [
            '2024-07-30 23:00:00',
            '2024-07-31 01:00:00',
            'Maintenance from 2024-07-30 23:00 to 2024-07-31 01:00',
        ];

        yield 'same time different days' => [
            '2024-07-30 10:00:00',
            '2024-07-31 10:00:00',
            'Maintenance from 2024-07-30 10:00 to 2024-07-31 10:00',
        ];
    }
}
