<?php

namespace Velis\Test\Mail;

use PHPUnit\Framework\TestCase;
use Velis\Mail\Parser;
use Velis\Mail\Storage\Maildir;
use Zend\Mail\Storage\Message as StorageMessage;

/**
 * Velis\Arrays test
 * @author Olek Procki <olo@velis.pl>
 */
class ParserTest extends TestCase
{
    /**
     * @var Parser
     */
    private $_parser;


    /**
     * @var Maildir
     */
    private $_storage;


    /**
     * Test initialization
     */
    protected function setUp(): void
    {
        $this->_parser  = new Parser();
        $this->_storage = new Maildir([
            'dirname' => MAIL_DIR,
        ]);
    }


    /**
     * Check multipart/signed parsing
     */
    public function testMultipartSigned()
    {
        $message = $this->_storage[1];
        $parsedMessage = $this->_parser->parse($message);

        $this->assertEquals('Re: system dla TP', $message->subject);
        $this->assertGreaterThan(0, strlen($parsedMessage['contentTextPlain']));
    }


    /**
     * Retrieving attached email messages test
     *
     * @param int $messageIndex
     * @param int $expectedAttachmentCount
     *
     * @dataProvider provideTestDataForGetAttachedEmailsMethod
     */
    public function testGetAttachedEmails($messageIndex, $expectedAttachmentCount)
    {
        $message = $this->_storage[$messageIndex];
        $this->_parser->parse($message);

        $attachedEmails = $this->_parser->getAttachedEmails();
        $this->assertEquals($expectedAttachmentCount, count($attachedEmails));
    }


    public function provideTestDataForGetAttachedEmailsMethod()
    {
        return [
            [2, 1],
        ];
    }


    public function testGetAttachedEmailWithPolishCharactersInAttachmentName()
    {
        $message = $this->_storage[3];
        $this->_parser->parse($message);

        $attachedEmails = $this->_parser->getAttachedEmails();
        $this->assertEquals($attachedEmails[0]['filename'], 'Zażółć gęślą jaźń.eml');
    }


    /**
     * Tests broken content-type header in mail part (application/application/pdf -> application/pdf)
     */
    public function testBrokenAttachmentType()
    {
        $params = array();
        $params['file'] = MAIL_DIR . '/new/04-broken-attachment-content-type.eml';

        $message = new StorageMessage($params);
        $result = $this->_parser->getEncodingInformation($message->getPart(2));

        $this->assertEquals('application/pdf', $result['contentType']);
    }


    /**
     * Test for unescaped subject retrieving
     */
    public function testUnescapedSubject()
    {
        $this->assertEquals(
            'Błędy aplikacji [Singu 2.0]',
            $this->_parser->getSubjectFromHeaders(
                $this->_storage->getRawHeader(5)
            )
        );
    }


    /**
     * Check id header retrieving
     *
     * @param int    $messageIndex
     * @param string $expectedResult
     *
     * @dataProvider provideTestDataForGetMessageIdMethod
     */
    public function testGetMessageId($messageIndex, $expectedResult)
    {
        $actualResult = $this->_parser->getMessageId($this->_storage[$messageIndex]);

        $this->assertEquals($expectedResult, $actualResult);
    }


    public function provideTestDataForGetMessageIdMethod()
    {
        return [
            'multiple IDs in single message' => [6, '<3X6T46F8ARLQ6F6EDVNUT7T005KMFW30@869-SN2MPN2-350.465d.mgd.msft.net>'],
            'single header' => [1, '<53919B97.6020909@spnt.pl>'],
        ];
    }
}
