<?php

namespace Velis\Test;

use PHPUnit\Framework\TestCase;
use Psr\SimpleCache\InvalidArgumentException;
use Velis\App;
use Velis\Bpm\Ticket\Status;
use Velis\Lang;

/**
 * @author Konrad Choma <konrad.choma@velistech.com>
 */
class LangTest extends TestCase
{
    /**
     * {@inheritDoc}
     */
    protected function setUp(): void
    {
        App::$session = App::$di['session'];
        Lang::switchLanguage('en');
    }

    /**
     * @param string $translation
     * @param string $expectedResult
     * @throws InvalidArgumentException
     *
     * @dataProvider provideDataForGetHistoryTranslationMethod
     */
    public function testGetHistoryTranslation($translation, $expectedResult)
    {
        $actualResult = Lang::getHistoryTranslation($translation);
        static::assertEquals($expectedResult, $actualResult);
    }

    /**
     * @return array
     * @throws InvalidArgumentException
     */
    public function provideDataForGetHistoryTranslationMethod(): array
    {
        App::$session = App::$di['session'];
        Lang::switchLanguage('en');

        // language code separator
        $lc = chr(30);
        // language version separator
        $lv = chr(31);
        $simple = [
            ['{GENERAL_ADD}', Lang::get('GENERAL_ADD')],
            ['{TICKET_STATUS}', Lang::get('TICKET_STATUS')],
            ['{TICKET_STATUS_CLOSED}', Status::get('Closed')],
            ['{TICKET_STATUS_2}', Lang::get('TICKET_STATUS_2')],
            ['Test', 'Test'],
            ['Test {}', 'Test {}'],
            ['2021-04-22', strftime('%x', strtotime('2021-04-22'))],
            [
                '2021-04-22 12:01:01',
                trim(preg_replace('/:\d{2}($|\s)/', ' ', strftime('%x %X', strtotime('2021-04-22 12:01:01')))),
            ],
            ['Test } {GENERAL_ADD} {', 'Test } ' . Lang::get('GENERAL_ADD') . ' {']
        ];

        $multipleVersionsCase1 = (Lang::getLanguage() == 'pl')
                                ? Lang::get('GENERAL_ADD')
                                : ((Lang::getLanguage() == 'en')
                                    ? Lang::get('GENERAL_ADD_2')
                                    : '');
        $multipleVersionsCase2 = (Lang::getLanguage() == 'pl')
                                ? Lang::get('TICKET_STATUS')
                                : ((Lang::getLanguage() == 'en')
                                    ? Lang::get('TICKET_STATUS_2')
                                    : '');
        $complex = [
            [
                "[{pl{$lc}{GENERAL_ADD}{$lv}en{$lc}{GENERAL_ADD_2}}]",
                $multipleVersionsCase1
            ],
            [
                "Test [{pl{$lc}{GENERAL_ADD}{$lv}en{$lc}{GENERAL_ADD_2}}] [{pl{$lc}{TICKET_STATUS}{$lv}en{$lc}{TICKET_STATUS_2}}]",
                "Test {$multipleVersionsCase1} {$multipleVersionsCase2}"
            ]
        ];

        return array_merge($simple, $complex);
    }
}
