<?php

namespace Velis\Test\Filter;

use Velis\Filter\Washtml;
use Velis\TestCase;

/**
 * Velis\Filter test
 * @author Olek Procki <olo@velis.pl>
 */
class WashtmlTest extends TestCase
{
    private Washtml $_filter;

    /**
     * {@inheritDoc}
     */
    protected function setUp(): void
    {
        $filterConfig = [
            'htmlElements' => ['style','head','body','html'],
            'htmlAttribs'    => null,
            'ignoreElements' => null,
            'voidElements'   => null,
            'base_url'       => 'velis.pl',
        ];

        $this->_filter = new Washtml($filterConfig);
    }

    /**
     * Test the elimination of some XSS vulnerabilities
     */
    public function testXss3()
    {
        $html = '<p><a href="data:text/html,&lt;script&gt;alert(document.cookie)&lt;/script&gt;">Firefox</a>'
            .'<a href="vbscript:alert(document.cookie)">Internet Explorer</a></p>';

        $washed = $this->_filter->filter($html);

        static::assertDoesNotMatchRegularExpression('/data:text/', $washed, 'Remove data:text/html links');
        static::assertDoesNotMatchRegularExpression('/vbscript:/', $washed, 'Remove vbscript: links');
    }

    /**
     * Test fixing of invalid href
     */
    public function testHref()
    {
        $html = "<p><a href=\"\nhttp://test.com\n\">Firefox</a>";
        $washed = $this->_filter->filter($html);

        static::assertMatchesRegularExpression('|href="http://test.com">|', $washed, 'Link href with newlines');
    }

    /**
     * Test handling HTML comments
     */
    public function testComments()
    {
        $html = '<!--[if gte mso 10]><p>p1</p><!--><p>p2</p>';
        $washed = $this->_filter->filter($html);

        static::assertEquals('<!-- node type 8 --><html><body><p>p2</p></body></html>', $washed, 'HTML conditional comments');

        $html = '<!--TestCommentInvalid><p>test</p>';
        $washed = $this->_filter->filter($html);

        static::assertEquals('<html><body><p>test</p></body></html>', $washed, 'HTML invalid comments');

        $html = '<p>para1</p><!-- comment --><p>para2</p>';
        $washed = $this->_filter->filter($html);

        static::assertEquals('<html><body><p>para1</p><!-- node type 8 --><p>para2</p></body></html>', $washed, 'HTML comments - simple comment');
    }

    /**
     * Test fixing of invalid self-closing elements
     */
    public function testSelfClosing()
    {
        $html = '<textarea>test';
        $washed = $this->_filter->filter($html);

        static::assertMatchesRegularExpression('|<textarea>test</textarea>|', $washed, 'Self-closing textarea');
    }

    /**
     * Test fixing of invalid closing tags
     */
    public function testClosingTagAttrs()
    {
        $html = "<a href=\"http://test.com\">test</a href>";
        $washed = $this->_filter->filter($html);

        static::assertMatchesRegularExpression('|</a>|', $washed, 'Invalid closing tag');
    }

    /**
     * Test color style handling
     */
    public function testColorStyle()
    {
        $html = "<p style=\"font-size: 10px; color: rgb(241, 245, 218)\">a</p>";
        $washed = $this->_filter->filter($html);

        static::assertMatchesRegularExpression('|color: rgb\(241, 245, 218\)|', $washed, 'Color style');
        static::assertMatchesRegularExpression('|font-size: 10px|', $washed, 'Font-size style');
    }

    /**
     * Test style item fixes
     */
    public function testStyleWash()
    {
        $html = "<p style=\"height: 10\">a</p>";
        $washed = $this->_filter->filter($html);

        static::assertMatchesRegularExpression('|height: 10px|', $washed, 'Fixed height units');
    }
}