<?php

namespace Velis\Test\Api\Query;

use PHPUnit\Framework\TestCase;
use Velis\Api\Query\SortOrderParser;
use Velis\Filter;

class SortOrderParserTest extends TestCase
{
    /**
     * @dataProvider provideDataForParseMethod
     */
    public function testParse(Filter $filter, string $expected, array $allowedFields = [], array $mappedFields = []): void
    {
        $parser = new SortOrderParser($filter, $allowedFields, $mappedFields);
        $sortOrder = $parser->parse();

        $this->assertEquals($expected, $sortOrder);
    }

    /**
     * @return array<int,array<int,mixed>>
     */
    public function provideDataForParseMethod(): array
    {
        return [
            [
                new Filter([
                    'sort_by' => 'name:asc,age:desc',
                ]),
                'name ASC, age DESC',
            ],
            [
                new Filter([
                    'sort_by' => 'name:asc',
                ]),
                'name ASC',
            ],
            [
                new Filter([
                    'sort_by' => 'name:desc',
                ]),
                'name DESC',
            ],
            [
                new Filter([
                    'sort_by' => 'name',
                ]),
                'name ASC',
            ],
            [
                new Filter([
                    'sort_by' => '',
                ]),
                '',
            ],
            [
                new Filter([]),
                '',
            ],
            [
                new Filter([
                    'sort_by' => 'name:asc,age:desc',
                ]),
                'name ASC',
                ['name'],
            ],
            [
                new Filter([
                    'sort_by' => 'name:asc,age:desc',
                ]),
                'name ASC, age DESC',
                ['name', 'age'],
            ],
            [
                new Filter([
                    'sort_by' => 'name:asc,age:desc',
                ]),
                '',
                ['surname'],
            ],
            [
                new Filter([
                    'sort_by' => 'ticket_status.name:asc,age:desc',
                ]),
                'app.get_dictionary_name(\'app.ticket_status_tab\', ticket_status_id) ASC',
                ['ticket_status.name'],
                ['ticket_status.name' => 'app.get_dictionary_name(\'app.ticket_status_tab\', ticket_status_id)'],
            ]
        ];
    }

    /**
     * @dataProvider provideDataForParseIntoArrayMethod
     */
    public function testParseIntoArray(Filter $filter, array $expected, array $allowedFields = [], array $mappedFields = []): void
    {
        $parser = new SortOrderParser($filter, $allowedFields, $mappedFields);
        $orderArray = $parser->parseIntoArray();

        $this->assertEquals($expected, $orderArray);
    }

    /**
     * @return array<int,array<int,mixed>>
     */
    public function provideDataForParseIntoArrayMethod(): array
    {
        return [
            [
                new Filter([
                    'sort_by' => 'name:asc,age:desc',
                ]),
                [
                    'name' => 'ASC',
                    'age' => 'DESC',
                ],
            ],
            [
                new Filter([
                    'sort_by' => 'name:asc',
                ]),
                [
                    'name' => 'ASC',
                ],
            ],
            [
                new Filter([
                    'sort_by' => 'name:desc',
                ]),
                [
                    'name' => 'DESC',
                ],
            ],
            [
                new Filter([
                    'sort_by' => 'name',
                ]),
                [
                    'name' => 'ASC',
                ],
            ],
            [
                new Filter([
                    'sort_by' => '',
                ]),
                [],
            ],
            [
                new Filter([]),
                [],
            ],
            [
                new Filter([
                    'sort_by' => 'name:asc,age:desc',
                ]),
                [
                    'name' => 'ASC',
                ],
                ['name'],
            ],
            [
                new Filter([
                    'sort_by' => 'name:asc,age:desc',
                ]),
                [
                    'name' => 'ASC',
                    'age' => 'DESC',
                ],
                ['name', 'age'],
            ],
            [
                new Filter([
                    'sort_by' => 'name:asc,age:desc',
                ]),
                [],
                ['surname'],
            ],
            [
                new Filter([
                    'sort_by' => 'ticket_status.name:asc,age:desc',
                ]),
                [
                    'app.get_dictionary_name(\'app.ticket_status_tab\', ticket_status_id)' => 'ASC',
                ],
                ['ticket_status.name'],
                ['ticket_status.name' => 'app.get_dictionary_name(\'app.ticket_status_tab\', ticket_status_id)'],
            ]
        ];
    }
}
