<?php

namespace Velis\Bpm\Ticket;

use RuntimeException;
use InvalidArgumentException;
use Velis\Arrays;
use Velis\Model\DataObject;
use Velis\Model\Cacheable;
use Velis\Filter;
use Velis\Lang;
use Velis\Output;

/**
 * Ticket classification model
 *
 * @author Bartosz Izdebski <bartosz.izdebski@velis.pl>
 * @author Olek Procki <olo@velis.pl>
 */
class Classification extends DataObject implements Cacheable
{
    /**
     * Buffer for not cached model type (must be redeclared in Cachable class)
     * @var \Velis\Model\Hierarchical[]
     */
    protected static $_source;


    /**
     * List sort options
     */
    const ORDER_NAME     = 'name_pl';
    const ORDER_ID       = 'ticket_classification_id DESC';


    /**
     * Returns related sql table name
     * @return string
     */
    protected function _getTableName()
    {
        return 'app.ticket_classification_tab';
    }


    /**
     * Returns classification name
     * @return string
     */
    public function getName()
    {
        return $this->getTranslatedName();
    }


    /**
     * Returns classification description
     * @return string
     */
    public function getDescription($default = false)
    {
        if (isset($this['description_' . Lang::getLanguage()])) {
            return $this['description_' . Lang::getLanguage()];
        } elseif ($default) {
            return Lang::get('GENERAL_NO_DESCRIPTION');
        } else {
            return $this['description'];
        }
    }

    /**
     * {@inheritDoc}
     */
    public static function listCached($objectId = null)
    {
        $classifications = parent::listCached();
        return Arrays::sortLocale($classifications);
    }


    /**
     * Remove classification if possible
     * @return bool
     */
    public function remove()
    {
        return parent::_remove();
    }

    /**
     * Check if it's an internal classification
     */
    public function isInternal(): bool
    {
        return $this['internal_classification'] == 1;
    }

    /**
     * @return string
     */
    public function getIcon($baseNameOnly = false)
    {
        if (!$baseNameOnly) {
            return $this['icon'] ? $this['icon'] : strtolower($this->id()) . '.png';
        }

        $output = $this['icon'];

        if ($output === null) {
            return $output;
        }

        $output = str_replace('fad', '', $output);

        $parts = explode(' ', $output);

        if (count($parts) > 1) {
            $output = $parts[1];
        }

        if (strpos($output, 'fa-') === 0) {
            $output = substr($output, 3);
        }

        return Output::toPascalCase($output);
    }


    /**
     * @return string
     */
    public function getMarker()
    {

        return $this['marker'] ? $this['marker'] : null;
    }


    /**
     * Returns classification list
     *
     * @param int $page
     * @param array|ArrayObject $params
     * @param string $order
     * @param int $limit
     * @param string|array $fields
     *
     * @return Velis\Bpm\Ticket\Classification[]
     */
    public static function getList($page = 1, $params = null, $order = self::ORDER_NAME, $limit = self::ITEMS_PER_PAGE, $fields = null)
    {
        return parent::getList($page, $params, $order, $limit, $fields);
    }


    /**
     * Save method disabled
     * @throws \RuntimeException always
     */
    public function save($updateObjectId = false)
    {
        throw new RuntimeException('Błędne wywołanie metody zapisu - użyj metod add()/modify()');
    }


    /**
     * Add classification
     *
     * @param bool $updateObjectId
     * @return \Velis\Bpm\Ticket|Classification
     *
     * @throws \InvalidArgumentException
     */
    public function add($updateObjectId = false)
    {
        if (!Filter::validateAcronym($this->id())) {
            throw new InvalidArgumentException('Niewłaściwy format ID typu (dozwolone są tylko litery/cyfry)');
        }
        return parent::add($updateObjectId);
    }
}
