<?php

namespace Velis\User\Absence;

use Velis\Report as ReportModel;
use Velis\Filter;
use Velis\User\Absence;

/**
 * Absences report
 * @author Bartosz Izdebski <bartosz.izdebski@velis.pl>
 */
class Report extends ReportModel
{

    protected $_firstDay;
    protected $_lastDay;

    protected $_lastCol;
    protected $_dates = array();
    protected $_users = array();


    /**
     * Report constructor
     *
     * @param array|ArrayObject $params
     * @throws \InvalidArgumentException
     */
    public function __construct($params)
    {
        parent::__construct($params);

        if (!Filter::validateDate($this->_params['start_date']) || !Filter::validateDate($this->_params['end_date'])) {
            throw new \InvalidArgumentException('Nie podano poprawnych dat dla raportu!');
        }

        if ($this->_params['end_date'] < $this->_params['start_date']) {
            throw new \InvalidArgumentException('Data końca nie może być wcześniejsza od daty startowej!');
        }

        $this->_firstDay = date('Y-m-d', strtotime('first day of this month', strtotime($this->_params['start_date'])));
        $this->_lastDay  = date('Y-m-d', strtotime('last day of this month', strtotime($this->_params['end_date'])));

        $this->_params['start_date'] = $this->_firstDay;
        $this->_params['end_date']   = $this->_lastDay;

        $date = $this->_firstDay;

        while ($date <= $this->_lastDay) {
            $dates[date("Y", strtotime($date))][date("m", strtotime($date))][] = $date;

            $date = date("Y-m-d", strtotime("+1 day", strtotime($date)));
        }

        $this->_dates = $dates;
    }


    /**
     * Returns month data
     * @return array
     */
    public function getYears()
    {
        return array_keys($this->_dates);
    }

    /**
     * Returns month data
     * @param int $monts
     * @return array
     */
    public function getMonths($year)
    {
        return array_keys($this->_dates[$year]);
    }

    /**
     * Returns month's days
     * @param int $year
     * @param int $month
     * @return array
     */
    public function getDays($year, $month)
    {
        return $this->_dates[$year][$month];
    }

    /**
     * Returns absences users
     * @return array
     */
    public function getAbsenceUsers()
    {
        return $this->_users;
    }


    /**
     * Prepares report data
     * @return array
     */
    public function prepare(array $params = array())
    {
        $outputData = $this->_dates;

        $absenceParams = array_merge(
            $params,
            array(
                        'archive' => 1,
                        'date_from' => $this->_params['start_date'],
                        'date_to' => $this->_params['end_date'],
                        'toAccept' => $this->_params['toAccept']
                    )
        );

        $absences = Absence::listAll($absenceParams);

        foreach ($absences as $absence) {
            $date = $absence['date_from'];

            while ($date <= $absence['date_to']) {
                $outputData[date("Y", strtotime($date))][date("m", strtotime($date))][date("Y-m-d", strtotime($date))][$absence['user_id']] = $absence;

                $this->_users[$absence['user_id']] = $absence['user_id'];

                $date = date("Y-m-d", strtotime("+1 day", strtotime($date)));
            }
        }

        return $outputData;
    }
}
