<?php

namespace Velis\Test\Mvc;

use Exception;
use PHPUnit\Framework\TestCase;
use Velis\App;
use Velis\Mvc\Router;

/**
 * @author Jan Małysiak <jan.malysiak@velis.pl>
 */
class RouterFunctionalTest extends TestCase
{
    /**
     * {@inheritDoc}
     */
    protected function setUp(): void
    {
        $_SERVER['SERVER_NAME'] = 'cafm.velis.pl';
        App::$session = App::$di['session'];
    }

    /**
     * @param string $uri
     * @param string $expectedModuleName
     * @param string $expectedControllerName
     * @param string $expectedActionName
     * @param $expectedParams
     * @throws Exception
     *
     * @dataProvider provideTestDataForApiRoutes
     */
    public function testApiRoutes(string $uri, string $expectedModuleName, string $expectedControllerName, string $expectedActionName, $expectedParams)
    {
        $_SERVER['REQUEST_URI'] = $uri;

        /** @var Router $router */
        $router = App::$di['router'];

        $router->init();
        $router->handle($uri);

        $actualModuleName = $router->getModuleName();
        $actualControllerName = $router->getControllerName();
        $actualActionName = $router->getActionName();
        $actualParams = $router->getParams();

        static::assertEquals($expectedModuleName, $actualModuleName);
        static::assertEquals($expectedControllerName, $actualControllerName);
        static::assertEquals($expectedActionName, $actualActionName);
        static::assertEquals($expectedParams, $actualParams);
    }

    /**
     * @return array
     */
    public function provideTestDataForApiRoutes(): array
    {
        return [
            ['/api/v1/ticket/vendor-ticket-list', 'Ticket', 'rest', 'VendorTicketList', ['version' => 1]],
            ['/api/v1/ticket/123456', 'Ticket', 'rest', 'get', ['id' => '123456', 'version' => 1]],
            ['/api/ticket/123456', 'Ticket', 'rest', 'get', ['id' => '123456', 'version' => '']],
            ['/api/v1/ticket', 'Ticket', 'rest', 'list', ['version' => 1]],
            ['/api/ticket', 'Ticket', 'rest', 'list', ['version' => '']],
        ];
    }

    /**
     * @param string $url
     * @param bool $expectedResult
     * @throws Exception
     *
     * @dataProvider provideTestDataForValidateUrlMethod
     * @runInSeparateProcess
     */
    public function testValidateUrl(string $url, bool $expectedResult)
    {
        /** @var Router $router */
        $router = App::$di['router'];
        $router->init();

        $actualResult = $router->validateUrl($url);

        static::assertEquals($expectedResult, $actualResult);
    }

    /**
     * @return array
     */
    public function provideTestDataForValidateUrlMethod(): array
    {
        return [
            ['/foo/bar', false],
            ['/tickets', App::$config->settings->defaultLanguage === 'en'],
            ['/zgloszenia', App::$config->settings->defaultLanguage === 'pl'],
            ['/building/ticket/list', true],
        ];
    }

    /**
     * {@inheritDoc}
     */
    protected function tearDown(): void
    {
        $_SERVER['SERVER_NAME'] = null;
    }
}
