<?php

namespace Velis\Test\Bpm\ErrorNotification;

use DateTime;
use DateTimeZone;
use PHPUnit\Framework\TestCase;
use Velis\Bpm\ErrorNotification\ErrorLogLineParser;

/**
 * @author Jan Małysiak <jan.malysiak@velis.pl>
 */
class ErrorLogLineParserTest extends TestCase
{
    /**
     * @var ErrorLogLineParser
     */
    private $parser;

    /**
     * {@inheritDoc}
     */
    protected function setUp(): void
    {
        $this->parser = new ErrorLogLineParser();
    }

    /**
     * @param string $line
     * @param $expectedResult
     *
     * @dataProvider provideTestData
     */
    public function testParseLine(string $line, $expectedResult)
    {
        $actualResult = $this->parser->parseLine($line);

        static::assertEquals($expectedResult, $actualResult);
    }

    public function provideTestData()
    {
        return [
            'line without time' => ['Message: The file "file/post-files/2021/0000020604_l" was not found', null],
            'line with time to be skipped' => ['[19-Mar-2021 21:15:14 America/New_York] PHP Stack trace:', null],
            'deprecated line' => [
                '[19-Mar-2021 21:15:14 America/New_York] PHP Deprecated:  Velis\Exception',
                [
                    'datetime' => new DateTime('19-Mar-2021 21:15:14', new DateTimeZone('America/New_York')),
                    'type' => 'Deprecated',
                    'message' => 'Velis\Exception',
                    'server' => null,
                ],
            ],
            'fatal error line' => [
                '[15-Mar-2021 10:24:44 America/Los_Angeles] PHP Fatal error:  Uncaught TypeError: Argument 1 passed to Velis\Output::formatTime() must be of the type float, null given, called in /home/devel/www/htdocs/module/Application/view/plugin/modifier.format_time.php on line 15 and defined in /home/devel/www/htdocs/lib/Velis/Output.php:278',
                [
                    'datetime' => new DateTime('15-Mar-2021 10:24:44', new DateTimeZone('America/Los_Angeles')),
                    'type' => 'error',
                    'message' => 'Uncaught TypeError: Argument 1 passed to Velis\Output::formatTime() must be of the type float, null given, called in /home/devel/www/htdocs/module/Application/view/plugin/modifier.format_time.php on line 15 and defined in /home/devel/www/htdocs/lib/Velis/Output.php:278',
                    'server' => null,
                ],
            ],
            'deprecated line with three-part timezone' => [
                '[03-Sep-2020 14:35:08 America/Kentucky/Louisville] PHP Deprecated:  Velis\Exception',
                [
                    'datetime' => new DateTime('03-Sep-2020 14:35:08', new DateTimeZone('America/Kentucky/Louisville')),
                    'type' => 'Deprecated',
                    'message' => 'Velis\Exception',
                    'server' => null,
                ],
            ],
            'line prefixed with server name' => [
                'app-prologisam-01 || [03-Sep-2020 14:35:08 America/Kentucky/Louisville] PHP Deprecated:  Velis\Exception',
                [
                    'datetime' => new DateTime('03-Sep-2020 14:35:08', new DateTimeZone('America/Kentucky/Louisville')),
                    'type' => 'Deprecated',
                    'message' => 'Velis\Exception',
                    'server' => 'app-prologisam-01',
                ],
            ],
        ];
    }
}
