<?php

namespace Velis\Sms;

use Velis\App;
use Velis\Sms;
use Velis\Output;
use Velis\Lang;

/**
 * SMS Api interface
 */
class SmsApi extends Sms implements SmsInterface
{
    protected static $_errorTypes = [
        8   => 'Błąd w odwołaniu SMS API (Prosimy zgłośić)',
        11  => 'Zbyt długa lub brak wiadomości lub ustawiono parametr nounicode i pojawiły się znaki specjalne w wiadomości.
                    Dla wysyłki VMS błąd oznacza brak pliku WAV lub treści TTS.',
        12  => 'Wiadomość zawiera ponad 160 znaków (gdy użyty parametr &single=1)',
        13  => 'Nieprawidłowy numer odbiorcy',
        14  => 'Nieprawidłowe pole nadawcy',
        17  => 'Nie można wysłać FLASH ze znakami specjalnymi',
        18  => 'Nieprawidłowa liczba parametrów',
        19  => 'Za dużo wiadomości w jednym odwołaniu',
        20  => 'Nieprawidłowa liczba parametrów IDX',
        21  => 'Wiadomość MMS ma za duży rozmiar (maksymalnie 300kB)',
        22  => 'Błędny format SMIL',
        23  => 'Błąd pobierania pliku dla wiadomości MMS lub VMS',
        24  => 'Błędny format pobieranego pliku',
        30  => 'Brak parametru UDH jak podany jest datacoding=bin',
        31  => 'Błąd konwersji TTS',
        32  => 'Nie można wysyłać wiadomości Eco, MMS i VMS na zagraniczne numery.',
        33  => 'Brak poprawnych numerów',
        40  => 'Brak grupy o podanej nazwie',
        41  => 'Wybrana grupa jest pusta (brak kontaktów w grupie)',
        50  => 'Nie można zaplanować wysyłki na więcej niż 3 miesiące w przyszłość',
        51  => 'Ustawiono błędną godzinę wysyłki, wiadomość VMS mogą być wysyłane tylko pomiędzy godzinami 8 a 22',
        52  => 'Za dużo prób wysyłki wiadomości do jednego numeru (maksymalnie 10 prób w przeciągu 60sek do jednego numeru)',
        101     => 'Niepoprawne lub brak danych autoryzacji SMS API. UWAGA! Hasło do API może różnić się od hasła do Panelu Klienta.',
        102     => 'Nieprawidłowy login lub hasło do SMS API',
        103     => 'Wykupiony pakiet SMS API się skończył. Prosimy o doładowanie, aby wiadomości SMS mogły być prawidłowo wysyłane przez aplikację',
        104     => 'Brak szablonu',
        105     => 'Błędny adres IP (włączony filtr IP dla interfejsu API)',
        200     => 'Nieudana próba wysłania wiadomości, prosimy ponowić odwołanie',
        201     => 'Wewnętrzny błąd systemu SMS API (prosimy zgłosić)',
        202     => 'Zbyt duża ilość jednoczesnych odwołań do serwisu, wiadomość nie została wysłana (prosimy odwołać się ponownie)',
        300     => 'Nieprawidłowa wartość pola points (przy użyciu pola points jest wymagana wartość 1)',
        301     => 'ID wiadomości nie istnieje',
        400     => 'Nieprawidłowy ID statusu wiadomości',
        999     => 'Wewnętrzny błąd systemu SMS API (prosimy zgłosić)'
    ];


    public function __construct()
    {
        $this->_config = [
            'base_url'    => App::$config->sms->baseUrl,
            'username'    => App::$config->sms->username,
            'password'    => App::$config->sms->password,
            'fast'        => (int)App::$config->sms->fast,
        ];

        if (!App::$config->sms->pro) {
            $this->_config['eco'] = 1;
        }
    }


    /**
     * Set SMS recipient number
     *
     * @param string $number
     * @return Velis_Model_Sms
     */
    public function setRecipientNumber($number)
    {
        $this->_recipientNumber = substr($number, -9);
        return $this;
    }


    /**
     * Set SMS recipient number
     *
     * @param string $content
     * @return Velis_Model_Sms
     */
    public function setContent($content)
    {
        if (!$this->_config['multipleSms']) {
            $content = substr($content, 0, Sms::SINGLE_SMS_LENGTH);
        }
        $this->_content = Output::stripPolishChars($content);
        return $this;
    }

    public function configIsCompleted(): bool
    {
        if (!App::$config->sms->baseUrl) {
            return false;
        }

        if (!App::$config->sms->username) {
            return false;
        }

        if (!App::$config->sms->password) {
            return false;
        }

        return true;
    }

    /**
     * Get API url for sending SMS
     *
     * @return string
     */
    protected function _getSendUrl($testSms = false)
    {
        $url = $this->_config['base_url'];

        $urlParams = [
            'username' => $this->_config['username'],
            'password' => md5($this->_config['password']),
            'to'       => $this->_recipientNumber,
            'message'  => $this->_content,
            'encoding' => 'utf-8',
            'fast'     => $this->_config['fast'],
            'eco'      => $this->_config['eco']
        ];

        if (App::$config->sms->passwordType === 'md5') {
            $urlParams['password'] = $this->_config['password'];
        }

        if (!$this->_config['multipleSms']) {
            $urlParams['single'] = 1;
        }

        if (App::$config->sms->test || $testSms) {
            $urlParams['test'] = 1;
        }

        $url .= '?' . http_build_query($urlParams);
        return $url;
    }

    /**
     * Sends sms
     * @return bool
     */
    public function send($test = false)
    {
        if (empty($this->_content)) {
            throw new \LogicException(Lang::get('GENERAL_SMS_NO_CONTENT'));
        }

        if (empty($this->_recipientNumber)) {
            throw new \LogicException(Lang::get('GENERAL_SMS_NO_PHONE'));
        }

        $sendStatus = file_get_contents($this->_getSendUrl($test));
        if (strpos($sendStatus, 'OK') === 0) {
            $this->log();
            return true;
        } else {
            $sendStatus = str_replace('ERROR:', '', $sendStatus);
            $message = array_key_exists($sendStatus, self::$_errorTypes) ? self::$_errorTypes[$sendStatus] : $sendStatus;

            throw new \InvalidArgumentException($message);
        }
    }
}
