<?php

namespace Velis;

use Exception;
use Velis\Model\BaseModel;
use Velis\Notification\Log;
use Velis\Sms\Amazon;
use Velis\Sms\SmsApi;

/**
 * SMS base
 * @author Pawel Robaczewski <pr@velis.pl>
 */
class Sms extends BaseModel
{
    /**
     * One sms maximal length
     *
     * @var integer
     */
    const SINGLE_SMS_LENGTH = 160;


    /**
     * SMS configuration
     *
     * @var array
     */
    protected $_config;

    /**
     * SMS content (text)
     *
     * @var string
     */
    protected $_content;


    /**
     * SMS backend
     */
    protected $_backend;


    /**
     * Execution time
     * @var float
     */
    protected $_execTime = 0;


    /**
     * SMS recipient phone number
     *
     * @var int
     */
    protected $_recipientNumber;


    /**
     * Constructor
     */
    public function __construct()
    {
        if (App::$config->sms->sendMethod == 'amazon') {
            $this->_backend = new Amazon();
        } else {
            $this->_backend = new SmsApi();
        }
        parent::__construct();
    }


    /**
     * Sends sms
     */
    public function send()
    {
        return $this->_backend->send();
    }


    /**
     * Set SMS recipient number
     *
     * @param string $number
     * @return Velis_Model_Sms
     */
    public function setRecipientNumber($number)
    {
        $this->_backend->setRecipientNumber(str_replace(' ', '', $number));
        return $this;
    }


    /**
     * Set multiple sms for content > 160 chars
     *
     */
    public function setMultipleSms()
    {
        $this->_config['multipleSms'] = true;
    }


    /**
     * Set SMS recipient number
     *
     * @param string $content
     * @return Velis_Model_Sms
     */
    public function setContent($content)
    {
        $this->_backend->setContent($content);
        return $this;
    }


    /**
     * Check if phone number is valid
     * @param string $number
     * @param string $message
     * @throws Exception
     */
    public static function checkNumber($number, $message = null)
    {
        $instance = new Sms();
        $instance->setRecipientNumber($number);
        $instance->setContent($message ?: 'test');

        $instance->send(true);
    }

    /**
     * @description Check if there is a configuration to use SMS selected sending method.
     * @return bool
     */
    public static function smsGateIsEnabled(): bool
    {
        $instance = (App::$config->sms->sendMethod == 'amazon') ? new Amazon() : new SmsApi();

        return $instance->configIsCompleted();
    }

    /**
     * Creates log entry for saving
     * @return Log
     * @throws Exception
     */
    public function log(): Log
    {
        $log = new Log(
            array(
                'subject' => $this->_content,
                'notification_method_id' => 'Sms',
                'body_txt' => $this->_content,
                'recipients' => $this->_recipientNumber,
                'is_sent' => 1,
                'send_date' => date('Y-m-d H:i:s')
            )
        );

        return $log->add(true);
    }
}
