<?php

namespace Velis\QuickReport;

use User\User;
use Velis\Acl\Role;
use Velis\Arrays;
use Velis\Model\Cacheable;
use Velis\Model\Hierarchical;

/**
 * Quick report category model
 * @author Bartosz Izdebski <bartosz.izdebski@velis.pl>
 * @author Olek Procki <olo@velis.pl>
 */
class Category extends Hierarchical implements Cacheable
{
    /**
     * Buffer for not cached model type (must be redeclared in Cachable class)
     * @var Hierarchical[]
     */
    protected static $_source;


    /**
     * users list
     * @var array
     */
    protected $_users;


    /**
     * roles list
     * @var array
     */
    protected $_roles;


    /**
     * Returns related table name
     * @return string
     */
    protected function _getTableName()
    {
        return 'app.quick_report_category_tab';
    }


    /**
     * {@inheritDoc}
     */
    public static function getNestedList(array $filter = null, array $predefinedList = null)
    {
        return parent::getNestedList($filter, $predefinedList);
    }


    /**
     * Returns category parent id
     * @return int
     */
    public function getParentId()
    {
        return $this->parent_quick_report_category_id;
    }


    /**
     * Get all user-based restrictions
     */
    public function getUsers()
    {
        if ($this->_users) {
            return $this->_users;
        } else {
            $query = "SELECT * FROM app.quick_report_category_user_tab WHERE quick_report_category_id = :quick_report_category_id";
            $params = array('quick_report_category_id' => $this->id());
            $users = Arrays::getColumn(self::$_db->getAll($query, $params), 'user_id');
            foreach ($users as $user) {
                $this->_users[$user] = User::get($user);
            }

            return $this->_users;
        }
    }


    /**
     * Remove all user-based restrictions
     *
     */
    public function removeUsers()
    {
        $params = array('quick_report_category_id' => $this->id());
        self::$_db->execDML("DELETE FROM app.quick_report_category_user_tab WHERE quick_report_category_id = :quick_report_category_id", $params);
    }


    /**
     * Add user-based restrictions
     *
     * @param array $users
     */
    public function addUsers(array $users)
    {
        foreach ($users as $user) {
            $params = array(
            'quick_report_category_id' => $this->id(),
            'user_id' => $user
            );
            self::$_db->insert('app.quick_report_category_user_tab', $params);
        }
    }


    /**
     * Remove user
     * @return int
     */
    public function removeUser($userId)
    {
        return self::$_db->execDML(
            'DELETE FROM app.quick_report_category_user_tab
             WHERE quick_report_category_id = :quick_report_category_id AND user_id = :user_id',
            array(
                'quick_report_category_id' => $this->id(),
                'user_id' => intval($userId)
            )
        );
    }


    /**
     * Add user
     */
    public function addUser($userId)
    {
        $this->addUsers(array($userId));
    }


    /**
     * Get all role-based restrictions
     */
    public function getRoles()
    {
        if ($this->_roles) {
            return $this->_roles;
        } else {
            $query = "SELECT * FROM app.quick_report_category_role_tab WHERE quick_report_category_id = :quick_report_category_id";
            $params = array('quick_report_category_id' => $this->id());
            $roles = Arrays::getColumn(self::$_db->getAll($query, $params), 'role_id');
            foreach ($roles as $role) {
                $this->_roles[$role] = Role::get($role);
            }
            return $this->_roles;
        }
    }


    /**
     * Remove all role-based restrictions
     */
    public function removeRoles()
    {
        self::$_db->execDML(
            "DELETE FROM app.quick_report_category_role_tab
             WHERE quick_report_category_id = :quick_report_category_id",
            array (
                'quick_report_category_id' => $this->id()
            )
        );
    }


    /**
     * Add role-based restrictions
     *
     * @param array $roles
     */
    public function addRoles(array $roles)
    {
        foreach ($roles as $role) {
            $params = array (
                'quick_report_category_id' => $this->id(),
                'role_id' => $role
            );
            self::$_db->insert('app.quick_report_category_role_tab', $params);
        }
    }


    /**
     * Remove role
     * @return int
     */
    public function removeRole($roleId)
    {
        return self::$_db->execDML(
            'DELETE FROM app.quick_report_category_role_tab
                 WHERE quick_report_category_id = :quick_report_category_id AND role_id = :role_id',
            array (
                'quick_report_category_id' => $this->id(),
                'role_id' => $roleId
            )
        );
    }


    /**
     * Add role
     */
    public function addRole($roleId)
    {
        $this->addRoles(array($roleId));
    }


    /**
     * Get categories by roleIds
     *
     * @param int[] $roleIds
     * @return array
     */
    public static function listByRoles($roleIds)
    {
        foreach ($roleIds as $roleId => $role) {
            $bindParams['role_id'][] = $roleId;
        }

        $sql = self::$_db->conditions($bindParams);
        $query = 'SELECT quick_report_category_id FROM app.quick_report_category_role_tab WHERE 1=1 ' . $sql;

        return self::$_db->getAll($query, $bindParams);
    }
}
