<?php

namespace Velis\Bpm\Ticket;

use Velis\Bpm\Email\Account;
use Velis\Lang;
use Velis\Mail\Protocol\SmtpPluginManager;
use Velis\Mail\Transport\Smtp;

/**
 * Notification messages model
 * @author Olek Procki <olo@velis.pl>
 */
class Notification extends \Velis\Notification
{
    const TICKET_CREATED                = 'ticket-created';
    const TICKET_POST                   = 'ticket-post';
    const TICKET_CHANGE                 = 'ticket-change';
    const TICKET_STATUS                 = 'ticket-status';
    const TICKET_OBSERVER_ADDED         = 'ticket-observer-added';
    const TICKET_OBSERVER_REMOVED       = 'ticket-observer-removed';
    const TICKET_DELEGATED_FROM         = 'ticket-delegated-from';
    const TICKET_DELEGATED_TO           = 'ticket-delegated-to';
    const TICKET_DELEGATED              = 'ticket-delegated';
    const TICKET_SERVICE_RATING         = 'ticket-service-rating';
    const TICKET_PROJECT_REGISTRATION   = 'ticket-project-registration';
    const OFFER_ACCEPTANCE_REQUEST      = 'offer-acceptance-request';
    const OFFER_ACCEPTED                = 'offer-accepted';
    const OFFER_REFUSED                 = 'offer-refused';



    /**
     * Constructor
     * @param \Velis\Bpm\Ticket $ticket
     */
    public function __construct(Ticket $ticket, $type)
    {
        $this->_type  = $type;

        parent::__construct();
        $this->ticket = $ticket;

        if (Ticket::hasField('email_account_id') && $ticket->email_account_id) {
            if ($account = Account::get($ticket->email_account_id)) {
                $config = array(
                    'auth'     => 'login',
                    'username' => $account->login,
                    'password' => $account->password,
                    'name'     => $account->outgoing_server
                );

                if ($account->outgoing_ssl) {
                    $config['ssl'] = $account->outgoing_ssl;
                }
                $this->_smtp = new Smtp($account->outgoing_server, $config);
                $this->_smtp->setPluginManager(new SmtpPluginManager());
            }
        }

        $this->setAdditionalHeaders([
            'X-Ticket-Id'       => $ticket->id(),
            'X-Ticket-Action'   => $type,
            'X-Ticket-Title'    => $ticket->title,
        ]);
    }


    /**
     * Return subject prefix
     * @return string
     */
    protected function _getSubjectPrefix()
    {
        return $this->ticket->getType() . ' ';
    }


    /**
     * Returns subject for notification email
     * @return string
     */
    protected function _getMailSubject()
    {
        $subject = $this->_getSubjectPrefix() . '#' . $this->ticket->id() . ': ' . $this->ticket->getTitle() . ' - ';

        switch ($this->_type) {
            case self::TICKET_CREATED:
                return $subject . mb_strtolower(Lang::get('TICKET_NEW'));

            case self::TICKET_POST:
                return $subject . mb_strtolower(Lang::get('GENERAL_NOTE'));

            case self::TICKET_CHANGE:
                return $subject . Lang::get('TICKET_UPDATE');

            case self::TICKET_OBSERVER_ADDED:
                return $subject . Lang::get('TICKET_BEEN_AN_OBSERVER');

            case self::TICKET_OBSERVER_REMOVED:
                return $subject . Lang::get('TICKET_REMOVAL_OF_OBSERVERS');

            case self::TICKET_STATUS:
                switch ($this->ticket->ticket_status_id) {
                    case Ticket::CONFIRMED:
                        return $subject . mb_strtolower(Lang::get('GENERAL_CONFIRMATION'));

                    case Ticket::CLOSED:
                        return $subject . Lang::get('GENERAL_CLOSING');
                }
                // no break
            case self::TICKET_DELEGATED_FROM:
                return $subject . Lang::get('TICKET_DELEGATE_TO_ANOTHER_PERSON');

            case self::TICKET_DELEGATED_TO:
                return $subject . Lang::get('TICKET_DELEGATE_TO_YOU');

            case self::TICKET_DELEGATED:
                return $subject . Lang::get('TICKET_DELEGATE_TO_ANOTHER_PERSON');

            case self::TICKET_SERVICE_RATING:
                return $subject . Lang::get('TICKET_REQUEST_FOR_ASSESSMENT');

            case self::OFFER_ACCEPTANCE_REQUEST:
                return Lang::get('TICKET_NEW_OFFER_IS_EXPECTED_TO_ACCEPT');

            case self::OFFER_ACCEPTED:
                return $subject . Lang::get('TICKET_PARTNER_ACCEPT_AN_OFFER');

            case self::OFFER_REFUSED:
                return $subject . Lang::get('TICKET_PARTNER_REJECTED_THE_OFFER');
        }
    }
}
