<?php

namespace Velis\Bpm\Product;

use InvalidArgumentException;
use Velis\Bpm\Vendor;
use Velis\Filter;
use Velis\Lang;
use Velis\Model\DataObject;
use Velis\Model\Cacheable;
use Velis\Output;

/**
 * ProductCategory model
 * @author Bartosz Izdebski <bartosz.izdebski@velis.pl>
 */
class Category extends DataObject implements Cacheable
{
    /**
     * @return string
     */
    protected function _getTableName()
    {
        return 'app.product_category_tab';
    }


    /**
     * Returns product category name
     * @return string
     */
    public function getName()
    {
        return $this->getTranslatedName();
    }


    /**
     * Validate PK format
     * @return bool
     *
     * @throws InvalidArgumentException
     */
    protected function _validate()
    {
        if (!Filter::validateAcronym($this->id(), true)) {
            throw new InvalidArgumentException(Lang::get('GENERAL_ALPHANUMERIC_CHARS_ONLY'));
        }
    }

    /**
     * Prepares object before saving
     * @return \Velis\Bpm\Product\Category
     */
    public function prepare()
    {
        $this['product_category_id'] = Output::acronym(
            $this->id(),
            Output::ACRO_UNDERSCORE
        );
        return $this;
    }


    /**
     * Returns product catagory list
     *
     * @param int $page
     * @param array|ArrayObject $params
     * @param string $order
     * @param int $limit
     * @param string|array $fields
     *
     * @return ProductCategory[]
     */
    public static function getList($page = 1, $params = null, $order = null, $limit = self::ITEMS_PER_PAGE, $fields = null)
    {

        if ($params['vendor_id']) {
            self::$_listConditions[] = "(EXISTS (SELECT 1 FROM app.vendor_product_category_tab vpc
                                                        WHERE vpc.vendor_id = :vendor_id AND vpc.product_category_id = app.product_category_tab.product_category_id))";

            self::$_listParams['vendor_id'] = $params['vendor_id'];
        }

        if ($params['project_ticket_id']) {
                self::$_listConditions[] = "EXISTS (
                    SELECT 1 FROM app.project_ticket_vendor_product_category_tab ptvpc
                    WHERE
                        ptvpc.project_ticket_id=:project_ticket_id
                        AND ptvpc.product_category_id = app.product_category_tab.product_category_id
                )";
                self::$_listParams['project_ticket_id'] = $params['project_ticket_id'];
        }

        return parent::getList($page, $params, $order, $limit, $fields);
    }


    /**
     * Returns vendor survey list
     * @return Velis\Bpm\Product\Category[]
     */
    public static function getNotAssigned($vendorId)
    {

        self::$_listConditions[] = "NOT EXISTS(SELECT 1 FROM app.vendor_product_category_tab vpc
                                        WHERE vpc.product_category_id = app.product_category_tab.product_category_id
                                        AND vpc.vendor_id = :vendor_id
                                    )";

        self::$_listParams['vendor_id'] = $vendorId;

        return parent::listAll();
    }


    /**
     *
     * @param \Velis\Bpm\Vender|string $vendor
     * @return \Velis\Bpm\Product\Category[]
     */
    public static function getAssigned($vendor)
    {
        $sql = 'SELECT * FROM app.vendor_product_category_tab WHERE vendor_id = :vendor_id';
        $params = array(
            'vendor_id' => $vendor instanceof Vendor ? $vendor->id() : $vendor,
        );

        $productCategories = array();
        foreach (self::$_db->getAll($sql, $params) as $category) {
            $productCategories[$category['product_category_id']] = $category;
        }
        return $productCategories;
    }


    /**
     * Modifies product category data
     *
     * @param bool $checkDiff
     * @return \Velis\Bpm\Product\Category
     */
    public function modify($checkDiff = false)
    {
        $this->_validate();
        return parent::modify($checkDiff);
    }


    /**
     * Adds new product category
     *
     * @param bool $updateObjectId
     * @return \Velis\Bpm\Product\Category
     */
    public function add($updateObjectId = false)
    {
        $this->_validate();
        return parent::add($updateObjectId);
    }


    /**
     * Removes product category
     * @return bool
     */
    public function remove()
    {
        return parent::_remove();
    }
}
